<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

require "config/db.php";
$user_id = (int)$_SESSION['user_id'];

/* =========================
   FETCH SAVINGS
========================= */
$sql = "
SELECT 
    s.expenses_id,
    s.amount,
    s.savings_type,
    s.note,
    s.savings_date,
    s.created_at
FROM savings s
WHERE s.user_id = ?
ORDER BY s.savings_date DESC
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$res = $stmt->get_result();

$savings = [];
$totalSavings = 0;

while ($r = $res->fetch_assoc()) {
    $totalSavings += (float)$r['amount'];
    $savings[] = $r;
}

$stmt->close();


/* =========================
   FETCH ACTIVE LOANS FROM SAVINGS
========================= */
$sql = "
SELECT IFNULL(SUM(amount),0) AS totalActiveLoans
FROM loans
WHERE user_id=? AND loan_source='savings' AND status='active'
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$totalActiveLoans = $stmt->get_result()->fetch_assoc()['totalActiveLoans'];
$stmt->close();

$netSavings = $totalSavings - $totalActiveLoans;
?>

<?php require "includes/head.php"; ?>
<?php include('includes/preloader.php'); ?>
<?php require "includes/navbar.php"; ?>

<div class="container my-4">

<h4 class="mb-3"><i class="bi bi-piggy-bank"></i> Savings</h4>

<!-- KPIs -->
<div class="row g-3 mb-4">
    <div class="col-md-4">
        <div class="p-3 rounded text-white" style="background:#0ea5e9">
            <i class="bi bi-wallet2"></i> Total Savings
            <h4>₵ <?= number_format($totalSavings,2) ?></h4>
        </div>
    </div>
    <div class="col-md-4">
        <div class="p-3 rounded text-white" style="background:#dc2626">
            <i class="bi bi-credit-card"></i> Active Loans from Savings
            <h4>₵ <?= number_format($totalActiveLoans,2) ?></h4>
        </div>
    </div>
    <div class="col-md-4">
        <div class="p-3 rounded text-white" style="background:#16a34a">
            <i class="bi bi-cash-stack"></i> Net Savings
            <h4>₵ <?= number_format($netSavings,2) ?></h4>
        </div>
    </div>
</div>

<!-- Add Saving Button -->
<div class="mb-3">
    <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addSavingModal">
        <i class="bi bi-plus-circle"></i> Add Saving
    </button>
</div>

<!-- Savings Table -->
<div class="card">
<div class="card-body table-responsive">
<table class="table table-sm table-bordered align-middle">
    <thead class="table-light">
        <tr>
            <th>Date</th>
            <th>Type</th>
            <th>Amount</th>
            <th>Note</th>
        </tr>
    </thead>
    <tbody>
        <?php if(empty($savings)): ?>
            <tr><td colspan="4" class="text-center text-muted">No savings found</td></tr>
        <?php else: foreach($savings as $s): ?>
            <tr>
                <td><?= date('M d, Y', strtotime($s['savings_date'])) ?></td>
                <td><?= htmlspecialchars(ucfirst($s['savings_type'])) ?></td>
                <td>₵ <?= number_format($s['amount'],2) ?></td>
                <td><?= htmlspecialchars($s['note']) ?></td>
            </tr>
        <?php endforeach; endif; ?>
    </tbody>
</table>
</div>
</div>

</div>

<!-- ADD SAVING MODAL -->
<div class="modal fade" id="addSavingModal" tabindex="-1">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">

      <div class="modal-header">
        <h5 class="fw-semibold">Add Saving</h5>
        <button class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <div class="modal-body">
        <form id="savingForm">

          <div class="row g-3 mb-3">
            <div class="col-md-4">
              <label class="form-label">Type</label>
              <select name="savings_type" class="form-select" required>
                <option value="cash">Cash</option>
                <option value="bank">Bank</option>
                <option value="other">Other</option>
              </select>
            </div>

            <div class="col-md-4">
              <label class="form-label">Amount</label>
              <input type="number" name="amount" id="savingAmount" class="form-control" placeholder="₵ 0.00" required>
            </div>

            <div class="col-md-4">
              <label class="form-label">Date</label>
              <input type="date" name="savings_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
            </div>

            <div class="col-md-12">
              <label class="form-label">Note</label>
              <textarea name="note" class="form-control" rows="2" placeholder="Optional note"></textarea>
            </div>
          </div>

          <div class="alert alert-danger d-none" id="netSavingWarning">
            Warning: Net saving after active loans will be negative!
          </div>

          <button type="submit" class="btn btn-success w-100" id="saveSavingBtn">
            <span class="stxt">Save Saving</span>
            <span class="spinner-border spinner-border-sm d-none"></span>
          </button>

        </form>
      </div>

    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

<script>
// Check net savings dynamically
document.addEventListener('DOMContentLoaded', () => {
    const savingInput = document.getElementById('savingAmount');
    const warningBox = document.getElementById('netSavingWarning');

    async function checkNetSaving(){
        const amount = parseFloat(savingInput.value) || 0;

        const res = await fetch('savings/get_active_loans.php');
        const data = await res.json();

        const net = amount - data.totalActiveLoans;

        if(net < 0){
            warningBox.classList.remove('d-none');
        } else {
            warningBox.classList.add('d-none');
        }
    }

    savingInput.addEventListener('input', checkNetSaving);

    document.getElementById('savingForm').addEventListener('submit', async e => {
        e.preventDefault();
        const btn = document.getElementById('saveSavingBtn');
        btn.disabled = true;
        btn.querySelector('.stxt').textContent = 'Saving...';
        btn.querySelector('.spinner-border').classList.remove('d-none');

        try {
            const formData = new FormData(e.target);
            const res = await fetch('expenses/add_saving.php', {method:'POST', body: formData});
            const data = await res.json();

            if(data.status==='success'){
                Swal.fire({icon:'success', title:'Saved!', timer:1500, showConfirmButton:false})
                .then(()=> location.reload());
            } else {
                Swal.fire('Error', data.message,'error');
            }
        } catch(err){
            Swal.fire('Error','Server error','error');
        } finally {
            btn.disabled = false;
            btn.querySelector('.stxt').textContent = 'Save Saving';
            btn.querySelector('.spinner-border').classList.add('d-none');
        }
    });
});
</script>
<?php $conn->close(); ?>