<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

require "config/db.php";
$user_id = (int)$_SESSION['user_id'];

$currentMonth = (int)date('m');
$currentYear  = (int)date('Y');

/* =========================
   FILTERS
========================= */
$where  = "WHERE user_id = ?";
$params = [$user_id];
$types  = "i";

$month       = $_GET['month'] ?? '';
$year        = $_GET['year'] ?? $currentYear;
$loan_source = $_GET['loan_source'] ?? '';

if ($month !== '') {
    $where .= " AND MONTH(loan_date) = ?";
    $params[] = (int)$month;
    $types   .= "i";
}

if ($year !== '') {
    $where .= " AND YEAR(loan_date) = ?";
    $params[] = (int)$year;
    $types   .= "i";
}

if ($loan_source !== '') {
    $where .= " AND loan_source = ?";
    $params[] = $loan_source;
    $types   .= "s";
}

/* =========================
   FETCH LOANS
========================= */
$sql = "
SELECT id, loan_source, amount, repaid_amount, status, description, loan_date
FROM loans
$where
ORDER BY loan_date DESC
";

$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

$loans = [];
$totalLoans = $totalRepaid = $totalOutstanding = 0;

while ($row = $res->fetch_assoc()) {
    $totalLoans += $row['amount'];
    $totalRepaid += $row['repaid_amount'];
    $totalOutstanding += ($row['amount'] - $row['repaid_amount']);
    $loans[] = $row;
}
$stmt->close();

/* =========================
   MONTHLY CHART DATA
========================= */
$chartLoans  = array_fill(1, $currentMonth, 0);
$chartRepaid = array_fill(1, $currentMonth, 0);

$stmt = $conn->prepare("
SELECT MONTH(loan_date) m, SUM(amount) loaned, SUM(repaid_amount) repaid
FROM loans
WHERE user_id=? AND YEAR(loan_date)=?
GROUP BY m
ORDER BY m
");
$stmt->bind_param("ii", $user_id, $year);
$stmt->execute();
$res = $stmt->get_result();

while ($r = $res->fetch_assoc()) {
    $chartLoans[(int)$r['m']]  = (float)$r['loaned'];
    $chartRepaid[(int)$r['m']] = (float)$r['repaid'];
}
$stmt->close();

$chartLabels = [];
for ($m=1; $m <= $currentMonth; $m++) {
    $chartLabels[] = date('M', mktime(0,0,0,$m,1));
}
?>

<?php require "includes/head.php"; ?>
<?php include('includes/preloader.php'); ?>
<?php require "includes/navbar.php"; ?>

<div class="container my-4">

<h4 class="mb-3"><i class="bi bi-cash-coin"></i> Loan Management</h4>

<!-- FILTER -->
<form class="row g-3 mb-4">
    <!-- Row 1: Month + Year -->
    <div class="col-6 col-md-3">
        <label class="form-label">Month</label>
        <select name="month" class="form-select">
            <option value="">All</option>
            <?php for ($m=1; $m<=$currentMonth; $m++): ?>
                <option value="<?= $m ?>" <?= ($month==$m)?'selected':'' ?>>
                    <?= date('F', mktime(0,0,0,$m,1)) ?>
                </option>
            <?php endfor; ?>
        </select>
    </div>

    <div class="col-6 col-md-3">
        <label class="form-label">Year</label>
        <input type="number" name="year" class="form-control" value="<?= $year ?>">
    </div>

    <!-- Row 2: Source -->
    <div class="col-12 col-md-3">
        <label class="form-label">Source</label>
        <select name="loan_source" class="form-select">
            <option value="">All</option>
            <?php foreach(['savings','bank','friends','other'] as $src): ?>
                <option value="<?= $src ?>" <?= ($loan_source==$src)?'selected':'' ?>>
                    <?= ucfirst($src) ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>

    <!-- Row 3: Buttons -->
    <div class="col-12 col-md-3 d-flex flex-column flex-md-row gap-2">
        <button class="btn btn-primary w-100">Filter</button>
        <a href="loans.php" class="btn btn-outline-secondary w-100">Reset</a>
    </div>
</form>


<!-- KPIs -->
<div class="row g-3 mb-4">
<?php
$cards = [
    ['Total Loans', $totalLoans, '#7c3aed', 'bi bi-cash-coin'],
    ['Total Repaid', $totalRepaid, '#16a34a', 'bi bi-wallet2'],
    ['Outstanding', $totalOutstanding, '#dc2626', 'bi bi-exclamation-circle']
];
?>
<div class="row g-3 mb-4">
<?php foreach($cards as [$title, $value, $color, $icon]): ?>
    <div class="col-6 col-md-4">
        <div class="p-3 rounded text-white d-flex align-items-center" style="background:<?= $color ?>">
            <i class="<?= $icon ?> me-2 fs-3"></i>
            <div>
                <?= $title ?>
                <h4 class="mb-0">₵ <?= number_format($value,2) ?></h4>
            </div>
        </div>
    </div>
<?php endforeach; ?>
</div>


<!-- ACTION BUTTONS -->
<div class="mb-3">
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#loanModal">
        <i class="bi bi-plus-circle"></i> Add Loan
    </button>
    <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#repayLoanModal">
        <i class="bi bi-cash-coin"></i> Repay Loan
    </button>
</div>

<!-- CHART -->
<div class="card mb-4">
<div class="card-body">
<canvas id="loanChart"></canvas>
</div>
</div>

<!-- TABLE -->
<div class="card">
<div class="table-responsive">
<table class="table table-bordered table-sm align-middle">
<thead class="table-light">
<tr>
<th>Date</th><th>Source</th><th>Loan</th><th>Repaid</th>
<th>Balance</th><th>Status</th><th>Description</th>
</tr>
</thead>
<tbody>
<?php if(!$loans): ?>
<tr><td colspan="7" class="text-center">No loans found</td></tr>
<?php endif; ?>
<?php foreach($loans as $l): ?>
<tr>
<td><?= date('M d, Y', strtotime($l['loan_date'])) ?></td>
<td><?= ucfirst($l['loan_source']) ?></td>
<td>₵ <?= number_format($l['amount'],2) ?></td>
<td class="text-success">₵ <?= number_format($l['repaid_amount'],2) ?></td>
<td class="text-danger">₵ <?= number_format($l['amount']-$l['repaid_amount'],2) ?></td>
<td>
<span class="badge bg-<?= $l['status']=='active'?'danger':'success' ?>">
<?= ucfirst($l['status']) ?>
</span>
</td>
<td><?= htmlspecialchars($l['description']) ?></td>
</tr>
<?php endforeach; ?>
</tbody>
</table>
</div>
</div>

</div>

<!-- ================= MODALS ================= -->

<!-- ADD LOAN MODAL -->
<div class="modal fade" id="loanModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">

      <div class="modal-header">
        <h5 class="modal-title"><i class="bi bi-currency-exchange me-2"></i>Add Loan</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <form id="loanForm" autocomplete="off">

        <div class="modal-body">

          <div class="mb-3">
            <label class="form-label">Category</label>
            <select name="category_id" class="form-select form-select-lg" required>
              <option value="">Select Category</option>
              <?php
                $catStmt = $conn->prepare("SELECT id, name FROM categories ORDER BY name");
                $catStmt->execute();
                $catRes = $catStmt->get_result();
                while($cat = $catRes->fetch_assoc()){
                  echo '<option value="'.$cat['id'].'">'.htmlspecialchars($cat['name']).'</option>';
                }
                $catStmt->close();
              ?>
            </select>
          </div>

          <div class="mb-3">
            <label class="form-label">Loan Source</label>
            <select name="loan_source" class="form-select form-select-lg" required>
              <option value="">Select Source</option>
              <option value="savings">Savings</option>
              <option value="friends">Friends</option>
              <option value="bank">Bank</option>
              <option value="other">Other</option>
            </select>
          </div>

          <div class="mb-3">
            <label class="form-label">Amount</label>
            <input type="number" name="amount" class="form-control" min="0.01" step="0.01" placeholder="₵ 0.00" required>
          </div>

          <div class="mb-3">
            <label class="form-label">Loan Date</label>
            <input type="date" name="loan_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
          </div>

          <div class="mb-2">
            <label class="form-label">Description (optional)</label>
            <textarea name="description" class="form-control" rows="2" placeholder="e.g. Loan for school fees"></textarea>
          </div>

        </div>

        <div class="modal-footer">
          <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-success"><i class="bi bi-check-circle me-1"></i>Save Loan</button>
        </div>

      </form>
    </div>
  </div>
</div>

<!-- REPAY LOAN MODAL -->
<div class="modal fade" id="repayLoanModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">

      <div class="modal-header">
        <h5 class="modal-title"><i class="bi bi-cash-coin me-2"></i>Repay Loan</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <form id="repayLoanForm">
        <div class="modal-body">

          <div class="mb-3">
            <label class="form-label">Select Loan</label>
            <select name="loan_id" id="loanSelect" class="form-select" required>
              <option value="">-- Choose a loan --</option>
              <?php
              $stmt = $conn->prepare("SELECT id, description, amount, repaid_amount FROM loans WHERE user_id=? AND status='active'");
              $stmt->bind_param("i",$user_id);
              $stmt->execute();
              $res = $stmt->get_result();
              while($loan = $res->fetch_assoc()):
                  $balance = $loan['amount'] - $loan['repaid_amount'];
              ?>
                <option value="<?= $loan['id'] ?>" data-balance="<?= $balance ?>">
                  <?= htmlspecialchars($loan['description'] ?: 'Loan #'.$loan['id']) ?> — Remaining: ₵ <?= number_format($balance,2) ?>
                </option>
              <?php endwhile; $stmt->close(); ?>
            </select>
          </div>

          <div class="mb-3">
            <label class="form-label">Amount to Repay</label>
            <input type="number" name="repay_amount" id="repayAmount" class="form-control" min="0.01" step="0.01" placeholder="₵ 0.00" required>
          </div>

        </div>

        <div class="modal-footer">
          <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-success">Repay</button>
        </div>
      </form>

    </div>
  </div>
</div>

<!-- ================= JS ================= -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Chart
new Chart(document.getElementById('loanChart'), {
    type: 'bar',
    data: {
        labels: <?= json_encode($chartLabels) ?>,
        datasets: [
            { label:'Loans', data: <?= json_encode(array_values($chartLoans)) ?> },
            { label:'Repaid', data: <?= json_encode(array_values($chartRepaid)) ?> }
        ]
    },
    options:{ responsive:true, scales:{ y:{ beginAtZero:true } } }
});

// Add Loan AJAX
$('#loanForm').submit(function(e){
    e.preventDefault();
    $.post('expenses/add_loan.php', $(this).serialize(), function(res){
        if(res.status==='success'){
            Swal.fire('Saved','Loan added','success').then(()=>location.reload());
        }else{
            Swal.fire('Error',res.message,'error');
        }
    },'json');
});

// Repay Loan AJAX
$('#loanSelect').change(function(){
    let balance = parseFloat($('option:selected',this).data('balance'));
    $('#repayAmount').attr('max',balance).val(balance);
});

$('#repayLoanForm').submit(function(e){
    e.preventDefault();
    $.post('expenses/repay_loan.php', $(this).serialize(), function(res){
        if(res.status==='success'){
            Swal.fire('Done','Loan repaid','success').then(()=>location.reload());
        }else{
            Swal.fire('Error',res.message,'error');
        }
    },'json');
});
</script>

<?php $conn->close(); ?>
