<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

require "config/db.php";
$user_id = $_SESSION['user_id'];

/* USER */
$stmt = $conn->prepare("SELECT name FROM users WHERE id=?");
$stmt->bind_param("i",$user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
$stmt->close();

/* =========================
   HELPER FUNCTION
========================= */
function total($conn, $uid, $where) {
    $q = $conn->prepare("
        SELECT IFNULL(SUM(amount),0) t
        FROM expenses
        WHERE user_id=? $where
    ");
    $q->bind_param("i", $uid);
    $q->execute();
    $r = (float)$q->get_result()->fetch_assoc()['t'];
    $q->close();
    return $r;
}

/* =========================
   ACTUAL EXPENSES
========================= */
$today = total($conn, $user_id, "AND DATE(expense_date)=CURDATE()");
$week  = total($conn, $user_id, "AND YEARWEEK(expense_date,1)=YEARWEEK(CURDATE(),1)");
$month = total(
    $conn,
    $user_id,
    "AND MONTH(expense_date)=MONTH(CURDATE())
     AND YEAR(expense_date)=YEAR(CURDATE())"
);




/* =========================
   PLANNED EXPENSES TOTALS
========================= */

/* =========================
   PLANNED TODAY (STRICT)
========================= */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(planned_amount), 0) AS total
    FROM planned_expenses
    WHERE user_id = ?
      AND plan_type = 'today'
      AND planned_date >= CURDATE()
      AND planned_date < CURDATE() + INTERVAL 1 DAY
");
$stmt->bind_param("i", $user_id);
$stmt->execute();

$plannedToday = (float) $stmt->get_result()->fetch_assoc()['total'];

$stmt->close();



/* =========================
   PLANNED EXPENSES (WEEK) — FIXED
========================= */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(planned_amount),0) AS total
    FROM planned_expenses
    WHERE user_id = ?
      AND plan_type = 'week'
      AND YEARWEEK(planned_date, 1) = YEARWEEK(CURDATE(), 1)
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$plannedWeek = (float)$stmt->get_result()->fetch_assoc()['total'];
/* =========================
   WEEKLY REMAINING
========================= */
$remainingWeek = max(0, $plannedWeek - $week);


$stmt->close();
/* =========================
   WEEKLY PLANNED — BY CATEGORY (SHOW ALL USED)
========================= */
$weeklyBreakdown = [];

$sql = "
SELECT
  c.name AS category,
  COALESCE(p.planned, 0) AS planned,
  COALESCE(e.used, 0) AS used
FROM categories c

/* Join planned expenses (might be NULL) */
LEFT JOIN (
  SELECT category_id, SUM(planned_amount) AS planned
  FROM planned_expenses
  WHERE user_id = ?
    AND plan_type = 'week'
    AND YEARWEEK(planned_date,1) = YEARWEEK(CURDATE(),1)
  GROUP BY category_id
) p ON p.category_id = c.id

/* Join actual expenses (might be NULL) */
LEFT JOIN (
  SELECT category_id, SUM(amount) AS used
  FROM expenses
  WHERE user_id = ?
    AND YEARWEEK(expense_date,1) = YEARWEEK(CURDATE(),1)
  GROUP BY category_id
) e ON e.category_id = c.id

/* Include any category that has either planned or used amount */
WHERE COALESCE(p.planned, 0) > 0 OR COALESCE(e.used, 0) > 0

ORDER BY c.name
";

$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $user_id, $user_id);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
    $row['remaining'] = max(0, $row['planned'] - $row['used']);
    $weeklyBreakdown[] = $row;
}
$stmt->close();



/* Total Planned This Month */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(planned_amount),0) AS total
    FROM planned_expenses
    WHERE user_id = ?
      AND MONTH(planned_date) = MONTH(CURDATE())
      AND YEAR(planned_date) = YEAR(CURDATE())
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$plannedMonth = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* Total Used This Month */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(amount),0) AS total
    FROM expenses
    WHERE user_id = ?
      AND MONTH(expense_date) = MONTH(CURDATE())
      AND YEAR(expense_date) = YEAR(CURDATE())
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$usedMonth = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* Monthly Remaining */
$remainingMonth = max(0, $plannedMonth - $usedMonth);



/* =========================
   KPI CALCULATIONS
========================= */
function percent(float $used, float $planned): int {
    return ($planned > 0) ? round(($used / $planned) * 100) : 0;
}
$todayPct = percent($today, $plannedToday);
$weekPct  = percent($week,  $plannedWeek);
$monthPct = percent($month, $plannedMonth);

/* DEFAULT SETTINGS */
$settings = [
    'occupation'  => '',
    'salary'      => '',
    'salary_date' => null
];

/* FETCH USER SETTINGS */
if (!empty($user_id)) {

    $stmt = $conn->prepare("
        SELECT occupation, salary, salary_date
        FROM user_settings
        WHERE user_id = ?
        LIMIT 1
    ");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();

    $res = $stmt->get_result();
    if ($res && $res->num_rows > 0) {
        $settings = array_merge($settings, $res->fetch_assoc());
    }

    $stmt->close();
}

/* ORDINAL HELPER (GLOBAL & SAFE) */
function ordinal(int $number): string
{
    if (!in_array($number % 100, [11, 12, 13])) {
        switch ($number % 10) {
            case 1: return $number . 'st';
            case 2: return $number . 'nd';
            case 3: return $number . 'rd';
        }
    }
    return $number . 'th';
}

$dailyLabels = [];
$dailyAmounts = [];

/* Initialize last 7 days */
for ($i = 6; $i >= 0; $i--) {
  $date = date('Y-m-d', strtotime("-$i days"));
  $dailyLabels[$date] = date('D', strtotime($date));
  $dailyAmounts[$date] = 0;
}

/* Fetch totals */
$stmt = $conn->prepare("
  SELECT expense_date, SUM(amount) total
  FROM expenses
  WHERE user_id = ?
    AND expense_date >= DATE_SUB(CURDATE(), INTERVAL 6 DAY)
  GROUP BY expense_date
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
  $dailyAmounts[$row['expense_date']] = (float)$row['total'];
}

//pei chart
$pieLabels = [];
$pieData   = [];
$pieColors = [];

$pieSql = "
  SELECT c.name, IFNULL(SUM(e.amount),0) AS total
  FROM expenses e
  LEFT JOIN categories c ON e.category_id = c.id
  WHERE e.user_id = ?
    AND YEARWEEK(e.expense_date, 1) = YEARWEEK(CURDATE(), 1)
  GROUP BY c.id, c.name
  HAVING total > 0
";

$stmt = $conn->prepare($pieSql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

$defaultColors = [
  '#2563EB', '#16A34A', '#F59E0B', '#DC2626', '#7C3AED',
  '#0EA5E9', '#DB2777', '#65A30D', '#EA580C', '#64748B'
];

$i = 0;
while ($row = $result->fetch_assoc()) {
  $pieLabels[] = $row['name'];
  $pieData[]   = $row['total'];
  $pieColors[] = $defaultColors[$i % count($defaultColors)];
  $i++;
}
/* =========================
   PLANNED vs USED (MONTHLY)
   FIXED (NO DUPLICATION)
========================= */

$sql = "
SELECT
  c.name AS category,
  IFNULL(p.planned, 0) AS planned,
  IFNULL(e.used, 0) AS used
FROM categories c

/* Planned totals per category */
LEFT JOIN (
  SELECT
    category_id,
    SUM(planned_amount) AS planned
  FROM planned_expenses
  WHERE user_id = ?
    AND plan_type = 'month'
    AND month = MONTH(CURDATE())
    AND year  = YEAR(CURDATE())
  GROUP BY category_id
) p ON p.category_id = c.id

/* Used totals per category */
LEFT JOIN (
  SELECT
    category_id,
    SUM(amount) AS used
  FROM expenses
  WHERE user_id = ?
    AND MONTH(expense_date) = MONTH(CURDATE())
    AND YEAR(expense_date)  = YEAR(CURDATE())
  GROUP BY category_id
) e ON e.category_id = c.id

WHERE p.planned IS NOT NULL OR e.used IS NOT NULL
ORDER BY c.name
";




$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $user_id, $user_id);
$stmt->execute();

$result = $stmt->get_result();
$plannedVsUsed = [];

/* =========================
   FETCH DATA FIRST
========================= */
while ($row = $result->fetch_assoc()) {
    $plannedVsUsed[] = $row;
}

/* =========================
   THEN CALCULATE percentage
========================= */
foreach ($plannedVsUsed as $k => $row) {

    $planned = (float)($row['planned'] ?? 0);
    $used    = (float)($row['used'] ?? 0);

    // Case 1: Used but no plan → force 100% (RED)
    if ($planned <= 0 && $used > 0) {
        $percentage = 100;
        $status     = 'over';
    }
    // Case 2: Planned exists
    elseif ($planned > 0) {
        $percentage = round(($used / $planned) * 100, 1);
        $status     = ($percentage > 100) ? 'over' : 'ok';
    }
    // Case 3: No plan & no usage
    else {
        $percentage = 0;
        $status     = 'ok';
    }

    $plannedVsUsed[$k]['percentage'] = $percentage;
    $plannedVsUsed[$k]['status']     = $status;
}

function incomeTotal($conn, $uid, $where) {
    $q = $conn->prepare("
        SELECT IFNULL(SUM(amount),0) t
        FROM salaries
        WHERE user_id=? $where
    ");
    $q->bind_param("i", $uid);
    $q->execute();
    $r = (float)$q->get_result()->fetch_assoc()['t'];
    $q->close();
    return $r;
}

/* =========================
   MONTHLY INCOME & CASHFLOW
========================= */

/* Total Income (This Month) */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(amount),0) total
    FROM salaries
    WHERE user_id = ?
      AND MONTH(income_date) = MONTH(CURDATE())
      AND YEAR(income_date)  = YEAR(CURDATE())
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$incomeMonth = (float)$stmt->get_result()->fetch_assoc()['total'];


/* Cashflow (Income - Expenses) */
$cashflowMonth = $incomeMonth - $month;

/* Status for styling */
$cashflowClass = ($cashflowMonth >= 0) ? 'positive' : 'negative';


/* =========================
   FETCH CATEGORIES
========================= */
$categories = [];

$stmt = $conn->prepare("
    SELECT id, name
    FROM categories
    ORDER BY name ASC
");
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $categories[] = $row;
}
$stmt->close();





/* =========================
   TOTAL LOANS (MONTH)
========================= */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(amount),0) AS total
    FROM loans
    WHERE user_id = ?
      AND MONTH(loan_date) = MONTH(CURDATE())
      AND YEAR(loan_date)  = YEAR(CURDATE())
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$totalLoansMonth = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();


/* =========================
   TOTAL SAVINGS
========================= */
$stmt = $conn->prepare("
    SELECT IFNULL(SUM(amount),0) AS total
    FROM savings
    WHERE user_id = ?
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$totalSavings = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* =========================
   LOANS FROM SAVINGS (CURRENT BALANCE)
========================= */
$stmt = $conn->prepare("
    SELECT 
        IFNULL(SUM(amount),0) - IFNULL(SUM(repaid_amount),0) AS total
    FROM loans
    WHERE user_id = ?
      AND loan_source = 'savings'
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$loansFromSavings = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* =========================
   CURRENT SAVINGS BALANCE
========================= */
$currentSavings = max(0, $totalSavings - $loansFromSavings);


// Fetch if user has any active loan
$hasLoan = false;

$stmt = $conn->prepare("
    SELECT 1
    FROM loans
    WHERE user_id = ?
      AND status = 'active'
    LIMIT 1
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->store_result();

$hasLoan = ($stmt->num_rows > 0);
$stmt->close();


?>

<?php include('includes/head.php'); ?>

<?php include('includes/preloader.php'); ?>
<?php include('includes/navbar.php'); ?>


<div class="container my-4">
  <h5 class="mb-4">Welcome, <?= htmlspecialchars($user['name']) ?> 👋</h5>

<!-- KPIs -->
<!-- KPIs -->
<div class="row g-3 mb-4">

<!-- Weekly Planned (Full Width) -->
<div class="col-12">
  <?php 
    // Determine card background class
    $cardBgClass = ($week > $plannedWeek) ? 'bg-danger text-white' : 'bg-primary text-white';
  ?>
  <div class="kpi planned p-3 rounded shadow-sm d-flex flex-column <?= $cardBgClass ?>">
    <div class="d-flex align-items-center mb-2">
      <i class="bi bi-calendar-week fs-5 me-2"></i>
      <small class="fs-6 fw-bold">Weekly Planned</small>
    </div>

    <h4 class="mt-1 mb-2 fs-5">
      ₵ <?= number_format($plannedWeek, 2) ?>
    </h4>

    <small class="d-block fs-7 <?= ($week > $plannedWeek) ? 'text-white' : 'text-white text-opacity-75' ?>">
      Spent: 
      <span class="<?= ($week > $plannedWeek) ? 'text-warning' : '' ?>">
        ₵ <?= number_format($week, 2) ?> (<?= $weekPct ?>%)
      </span>
    </small>

    <small class="d-block mb-2 fs-7 text-white text-opacity-75">
      Remaining: ₵ <?= number_format($remainingWeek, 2) ?>
    </small>

    <div class="small border-top border-white border-opacity-25 pt-2 mt-auto text-white text-opacity-75">
      <?php if (!empty($weeklyBreakdown)): ?>
        <?php foreach ($weeklyBreakdown as $row): ?>
          <div class="d-flex justify-content-between align-items-center mb-1 fs-7">
            <span class="text-truncate" style="max-width:60%;">
              <?= htmlspecialchars($row['category']) ?>
            </span>
            <span class="<?= ($row['used'] > $row['planned']) ? 'text-warning' : '' ?>">
              ₵<?= number_format($row['used'],2) ?> / ₵<?= number_format($row['planned'],2) ?>
            </span>
          </div>
        <?php endforeach; ?>
      <?php else: ?>
        <div class="text-center fs-7">No weekly plans yet</div>
      <?php endif; ?>
    </div>
  </div>
</div>


  <!-- Today -->
  <div class="col-6 col-md-4">
    <div class="kpi today p-3 rounded shadow-sm h-100 d-flex flex-column bg-success text-white">
      <div class="d-flex align-items-center mb-2">
        <i class="bi bi-calendar-day fs-5 me-2"></i>
        <small class="fs-6 fw-bold">Today</small>
      </div>

      <h4 class="mt-1 mb-2 fs-5">
        ₵ <?= number_format($today,2) ?>
      </h4>

      <small class="fs-7 text-white text-opacity-75">
        Planned ₵ <?= number_format($plannedToday,2) ?> • <?= $todayPct ?>%
      </small>
    </div>
  </div>




<!-- Total Planned (Month) -->
<div class="col-6 col-md-4">
  <div class="kpi planned p-3 rounded shadow-sm d-flex flex-column bg-info text-white">
    <div class="d-flex align-items-center mb-2">
      <i class="bi bi-clipboard2-check fs-5 me-2"></i>
      <small class="fs-6 fw-bold">Total Planned (Month)</small>
    </div>

    <h4 class="mt-1 mb-2 fs-5 text-white">
      ₵ <?= number_format($plannedMonth, 2) ?>
    </h4>

    <small class="d-block fs-7 text-white text-opacity-75 mb-2">
      Used: ₵ <?= number_format($usedMonth, 2) ?>  <!-- Total used amount -->
    </small>

    <div class="small border-top border-white border-opacity-25 pt-2 mt-auto text-white text-opacity-75">
      <?php if (!empty($monthlyBreakdown)): ?>
        <?php foreach ($monthlyBreakdown as $row): 
            $remaining = max(0, $row['planned'] - $row['used']); // Calculate remaining per category
        ?>
          <div class="d-flex justify-content-between align-items-center mb-1 fs-7">
            <span class="text-truncate" style="max-width:60%;">
              <?= htmlspecialchars($row['category']) ?>
            </span>
            <span class="<?= $row['used'] > $row['planned'] ? 'text-danger' : '' ?>">
              ₵<?= number_format($row['used'],2) ?> / ₵<?= number_format($row['planned'],2) ?>
              <?php if($remaining > 0): ?>
                (Remaining: ₵<?= number_format($remaining,2) ?>)
              <?php endif; ?>
            </span>
          </div>
        <?php endforeach; ?>
      <?php else: ?>
        <div class="text-center fs-7">No monthly plans yet</div>
      <?php endif; ?>
    </div>
  </div>
</div>




  <!-- Income -->
  <div class="col-6 col-md-4">
    <div class="kpi income p-3 rounded shadow-sm h-100 d-flex flex-column bg-success text-white">
      <div class="d-flex align-items-center mb-2">
        <i class="bi bi-cash-stack fs-5 me-2"></i>
        <small class="fs-6 fw-bold">Income (Month)</small>
      </div>

      <h4 class="mt-1 mb-2 fs-5">
        ₵ <?= number_format($incomeMonth, 2) ?>
      </h4>

      <small class="fs-7 text-white text-opacity-75">
        <?= date('F Y') ?>
      </small>
    </div>
  </div>

  <!-- Cashflow -->
  <div class="col-6 col-md-4">
    <div class="kpi p-3 rounded shadow-sm h-100 d-flex flex-column bg-danger text-white">
      <div class="d-flex align-items-center mb-2">
        <i class="bi bi-bar-chart-line fs-5 me-2"></i>
        <small class="fs-6 fw-bold">Cashflow (Month)</small>
      </div>

      <h4 class="mt-1 mb-2 fs-5">
        ₵ <?= number_format($cashflowMonth, 2) ?>
      </h4>

      <small class="fs-7 text-white text-opacity-75">
        Income ₵<?= number_format($incomeMonth,2) ?> • Spent ₵<?= number_format($month,2) ?>
      </small>
    </div>
  </div>
<!-- Total Loans (Month) -->
<div class="col-6 col-md-4">
  <div class="kpi loans p-3 rounded shadow-sm h-100 d-flex flex-column bg-danger text-white">

    <div class="d-flex align-items-center mb-2">
      <i class="bi bi-cash-stack fs-5 me-2"></i>
      <small class="fs-6 fw-bold">Loans (This Month)</small>
    </div>

    <h4 class="mt-1 mb-2 fs-5">
      ₵ <?= number_format($totalLoansMonth, 2) ?>
    </h4>

    <small class="fs-7 text-white text-opacity-75">
      Total borrowed this month
    </small>

  </div>
</div>


<!-- Current Savings Balance -->
<div class="col-6 col-md-4">
  <div class="kpi savings p-3 rounded shadow-sm h-100 d-flex flex-column bg-success text-white">

    <div class="d-flex align-items-center mb-2">
      <i class="bi bi-piggy-bank fs-5 me-2"></i>
      <small class="fs-6 fw-bold">Current Savings</small>
    </div>

    <h4 class="mt-1 mb-2 fs-5">
      ₵ <?= number_format($currentSavings, 2) ?>
    </h4>

    <small class="fs-7 text-white text-opacity-75">
      Savings − Loans from savings
    </small>

    <div class="small border-top border-white border-opacity-25 pt-2 mt-auto text-white text-opacity-75 fs-7">
      <div class="d-flex justify-content-between">
        <span>Total Saved</span>
        <span>₵<?= number_format($totalSavings, 2) ?></span>
      </div>
      <div class="d-flex justify-content-between">
        <span>Used for Loans</span>
        <span class="text-warning">
          − ₵<?= number_format($loansFromSavings, 2) ?>
        </span>
      </div>
    </div>

  </div>
</div>

</div>






<!-- ACTION BUTTONS -->
<?php include('includes/action_btn.php'); ?>

<!-- DAILY EXPENSES CHART -->
<div class="mt-4 mb-4">
  <div class="card shadow-sm">
    <div class="card-body">
      <h6 class="fw-semibold mb-3">Daily Expenses (Last 7 Days)</h6>

      <div style="height:220px;">
        <canvas id="dailyExpenseChart"></canvas>
      </div>
    </div>
  </div>
</div>

<div class="mt-4 mb-5">
  <div class="card shadow-sm">
    <div class="card-body">
      <h6 class="fw-semibold mb-3 text-center">
        Weekly Expenses by Category
      </h6>

      <div style="max-width:420px;margin:auto">
        <canvas id="categoryPie"></canvas>
      </div>
    </div>
  </div>
</div>

<!-- PLANNED VS USED -->
<div class="card shadow-sm mb-5">
  <div class="card-body">

    <h6 class="fw-semibold mb-3">Planned vs Used</h6>

    <div class="table-responsive">
      <table class="table table-sm align-middle table-bordered text-nowrap mb-0">

        <thead class="table-light">
          <tr>
            <th>Category</th>
            <th class="text-end">Planned</th>
            <th class="text-end">Used</th>
            <th style="width:140px">Usage</th>
            <th class="text-center">Action</th>
          </tr>
        </thead>

        <tbody>
        <?php foreach ($plannedVsUsed as $row): ?>

        <?php
          $pct    = $row['percentage'] ?? 0;
          $status = $row['status'] ?? 'ok';

          if ($status === 'over') {
              $barClass  = 'bg-danger';
              $textClass = 'text-danger';
          } else {
              $barClass  = 'bg-success';
              $textClass = 'text-success';
          }
        ?>

        <tr>
          <td><?= htmlspecialchars($row['category']) ?></td>

          <td class="text-end">
            ₵<?= number_format($row['planned'], 2) ?>
          </td>

          <td class="text-end">
            ₵<?= number_format($row['used'], 2) ?>
          </td>

          <td>
            <div class="progress mb-1" style="height:8px">
              <div class="progress-bar <?= $barClass ?>"
                   style="width: <?= min($pct,100) ?>%">
              </div>
            </div>
            <small class="<?= $textClass ?>"><?= $pct ?>%</small>
          </td>

          <td class="text-center">
            <button class="btn btn-sm btn-outline-primary"
                    onclick="viewExpenses('<?= htmlspecialchars($row['category']) ?>')">
              View
            </button>
          </td>
        </tr>

        <?php endforeach; ?>
        </tbody>

      </table>
    </div>

  </div>
</div>

<!--- Modals -->
<?php include('includes/modals.php'); ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>


<!--- Modals -->
<?php include('includes/modals_script.php'); ?>

<script>
const pieCtx = document.getElementById('categoryPie');

if (pieCtx) {
  new Chart(pieCtx, {
    type: 'pie',
    data: {
      labels: <?= json_encode($pieLabels) ?>,
      datasets: [{
        data: <?= json_encode($pieData) ?>,
        backgroundColor: <?= json_encode($pieColors) ?>,
        borderWidth: 1
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          position: 'bottom',
          labels: {
            boxWidth: 14,
            padding: 15
          }
        },
        tooltip: {
          callbacks: {
            label: function(ctx) {
              return ctx.label + ': ₵ ' + ctx.parsed.toLocaleString();
            }
          }
        }
      }
    }
  });
}
</script>

<script>
const ctx = document.getElementById('dailyExpenseChart');

new Chart(ctx, {
  type: 'bar',
  data: {
    labels: <?= json_encode(array_values($dailyLabels)) ?>,
    datasets: [{
      data: <?= json_encode(array_values($dailyAmounts)) ?>,
      backgroundColor: '#2563eb',
      borderRadius: 6,
      barPercentage: 0.6,
      categoryPercentage: 0.7
    }]
  },
  options: {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: { display: false },
      tooltip: {
        callbacks: {
          label: ctx => ` ₵ ${ctx.raw.toFixed(2)}`
        }
      }
    },
    scales: {
      y: {
        beginAtZero: true,
        ticks: {
          maxTicksLimit: 4,
          callback: v => '₵' + v
        },
        grid: {
          drawBorder: false,
          color: '#eee'
        }
      },
      x: {
        grid: { display: false }
      }
    }
  }
});
</script>



<script>
const realDate   = document.getElementById('expenseDate');
const textDate   = document.getElementById('expenseDateText');

// Format to human readable
function formatDate(val){
  const d = new Date(val);
  return d.toLocaleDateString(undefined,{
    weekday:'long',
    year:'numeric',
    month:'long',
    day:'numeric'
  });
}

// Set default
textDate.value = formatDate(realDate.value);

// Open picker when clicking text
textDate.addEventListener('click', () => {
  realDate.showPicker ? realDate.showPicker() : realDate.click();
});

// Update text when date changes
realDate.addEventListener('change', () => {
  textDate.value = formatDate(realDate.value);
});
</script>
  


<?php
$conn->close();
?>

</body>
</html>
