<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

require "config/db.php";
$user_id = (int)$_SESSION['user_id'];

/* =========================
   FILTER INPUTS
========================= */
$period      = $_GET['period']   ?? 'month';
$category_id = $_GET['category'] ?? '';

$month = $_GET['month'] ?? date('m');
$year  = $_GET['year']  ?? date('Y');

/* =========================
   CURRENT DATE RANGE
========================= */
switch ($period) {

    case 'day':
        $from = $to = date('Y-m-d');
        break;

    case 'week':
        $from = date('Y-m-d', strtotime('monday this week'));
        $to   = date('Y-m-d', strtotime('sunday this week'));
        break;

    case 'year':
        $from = "$year-01-01";
        $to   = "$year-12-31";
        break;

    default: // month
        $from = "$year-$month-01";
        $to   = date('Y-m-t', strtotime($from));
}

/* =========================
   PREVIOUS DATE RANGE
========================= */
switch ($period) {

    case 'day':
        $prevFrom = date('Y-m-d', strtotime('-1 day', strtotime($from)));
        $prevTo   = $prevFrom;
        break;

    case 'week':
        $prevFrom = date('Y-m-d', strtotime('-7 days', strtotime($from)));
        $prevTo   = date('Y-m-d', strtotime('-7 days', strtotime($to)));
        break;

    case 'year':
        $prevFrom = date('Y-01-01', strtotime('-1 year', strtotime($from)));
        $prevTo   = date('Y-12-31', strtotime('-1 year', strtotime($from)));
        break;

    default: // month
        $prevFrom = date('Y-m-01', strtotime('-1 month', strtotime($from)));
        $prevTo   = date('Y-m-t', strtotime($prevFrom));
}

/* =========================
   HUMAN LABELS
========================= */
$fromHuman     = date('M d, Y', strtotime($from));
$toHuman       = date('M d, Y', strtotime($to));
$prevFromHuman = date('M d, Y', strtotime($prevFrom));
$prevToHuman   = date('M d, Y', strtotime($prevTo));

$currentLabel  = "$fromHuman → $toHuman";
$previousLabel = "$prevFromHuman → $prevToHuman";

/* =========================
   FETCH CATEGORIES
========================= */
$categories = $conn->query("
    SELECT id, name
    FROM categories
    ORDER BY name
");

/* =========================
   TOTAL USED (CURRENT)
========================= */
$sql = "
    SELECT IFNULL(SUM(amount),0) total
    FROM expenses
    WHERE user_id = ?
      AND expense_date BETWEEN ? AND ?
";

$params = [$user_id, $from, $to];
$types  = "iss";

if ($category_id !== '') {
    $sql .= " AND category_id = ?";
    $params[] = $category_id;
    $types   .= "i";
}

$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$usedTotal = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* =========================
   TOTAL USED (PREVIOUS)
========================= */
$sqlPrev = "
    SELECT IFNULL(SUM(amount),0) total
    FROM expenses
    WHERE user_id = ?
      AND expense_date BETWEEN ? AND ?
";

$params = [$user_id, $prevFrom, $prevTo];
$types  = "iss";

if ($category_id !== '') {
    $sqlPrev .= " AND category_id = ?";
    $params[] = $category_id;
    $types   .= "i";
}

$stmt = $conn->prepare($sqlPrev);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$prevTotal = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* =========================
   TOTAL PLANNED
========================= */
$sql = "
    SELECT IFNULL(SUM(planned_amount),0) total
    FROM planned_expenses
    WHERE user_id = ?
      AND planned_date BETWEEN ? AND ?
      AND plan_type = ?
";

$params = [$user_id, $from, $to, $period];
$types  = "isss";

if ($category_id !== '') {
    $sql .= " AND category_id = ?";
    $params[] = $category_id;
    $types   .= "i";
}

$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$plannedTotal = (float)$stmt->get_result()->fetch_assoc()['total'];
$stmt->close();

/* =========================
   KPI CALCULATIONS
========================= */
$remaining = max(0, $plannedTotal - $usedTotal);

$percent = $plannedTotal > 0
    ? round(($usedTotal / $plannedTotal) * 100)
    : 0;

/* =========================
   EXPENSE LIST
========================= */
$sql = "
    SELECT
        e.id,
        e.expense_date,
        e.amount,
        e.description,
        c.name AS category
    FROM expenses e
    LEFT JOIN categories c ON c.id = e.category_id
    WHERE e.user_id = ?
      AND e.expense_date BETWEEN ? AND ?
";

$params = [$user_id, $from, $to];
$types  = "iss";

if ($category_id !== '') {
    $sql .= " AND e.category_id = ?";
    $params[] = $category_id;
    $types   .= "i";
}

$sql .= " ORDER BY e.expense_date DESC";

$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$expenses = $stmt->get_result();
$stmt->close();

/* =========================
   CATEGORY SUMMARY
========================= */
$stmt = $conn->prepare("
    SELECT
        c.id,
        c.name,
        IFNULL(SUM(e.amount),0) used
    FROM categories c
    LEFT JOIN expenses e
      ON e.category_id = c.id
     AND e.user_id = ?
     AND e.expense_date BETWEEN ? AND ?
    GROUP BY c.id
    HAVING used > 0
    ORDER BY used DESC
");

$stmt->bind_param("iss", $user_id, $from, $to);
$stmt->execute();
$catSummary = $stmt->get_result();
$stmt->close();

?>
<?php require "includes/head.php"; ?>
<?php include('includes/preloader.php'); ?>
<?php require "includes/navbar.php"; ?>

<div class="container my-4">

<h5 class="mb-3">Expenses Summary</h5>

<div class="card mb-3">
  <div class="card-body d-flex flex-wrap gap-2 justify-content-between align-items-center">

    <small class="text-muted">
      Showing <strong><?= $fromHuman ?></strong> → <strong><?= $toHuman ?></strong>
    </small>

<form id="filterForm" class="d-flex gap-2 flex-wrap">

  <!-- PERIOD -->
  <select name="period" class="form-select form-select-sm">
    <option value="day"   <?= $period==='day'?'selected':'' ?>>Today</option>
    <option value="week"  <?= $period==='week'?'selected':'' ?>>Week</option>
    <option value="month" <?= $period==='month'?'selected':'' ?>>Month</option>
    <option value="year"  <?= $period==='year'?'selected':'' ?>>Year</option>
  </select>

  <!-- MONTH -->
  <select name="month" class="form-select form-select-sm">
    <?php for($m=1;$m<=12;$m++):
      $val = str_pad($m,2,'0',STR_PAD_LEFT); ?>
      <option value="<?= $val ?>" <?= $month==$val?'selected':'' ?>>
        <?= date('F', mktime(0,0,0,$m,1)) ?>
      </option>
    <?php endfor; ?>
  </select>

  <!-- YEAR -->
  <select name="year" class="form-select form-select-sm">
    <?php for($y=date('Y')-5;$y<=date('Y')+1;$y++): ?>
      <option value="<?= $y ?>" <?= $year==$y?'selected':'' ?>>
        <?= $y ?>
      </option>
    <?php endfor; ?>
  </select>

  <!-- CATEGORY -->
  <select name="category" class="form-select form-select-sm">
    <option value="">All</option>
    <?php while($c=$categories->fetch_assoc()): ?>
      <option value="<?= $c['id'] ?>" <?= $category_id==$c['id']?'selected':'' ?>>
        <?= htmlspecialchars($c['name']) ?>
      </option>
    <?php endwhile; ?>
  </select>

</form>

  </div>
</div>


<!-- KPI -->
<div class="row g-3 mb-4">
  <div class="col-6 col-md-3">
    <div class="kpi planned">
      <small>Planned</small>
      <h4>₵ <?= number_format($plannedTotal,2) ?></h4>
    </div>
  </div>
  <div class="col-6 col-md-3">
    <div class="kpi used">
      <small>Used</small>
      <h4>₵ <?= number_format($usedTotal,2) ?></h4>
      <small><?= $percent ?>%</small>
    </div>
  </div>
  <div class="col-6 col-md-3">
    <div class="kpi remaining">
      <small>Remaining</small>
      <h4>₵ <?= number_format($remaining,2) ?></h4>
    </div>
  </div>
</div>

<!-- VIEW CATEGORIES MODAL -->
<div class="d-flex justify-content-end mb-4">
  <button
    class="btn btn-sm btn-outline-primary"
    data-bs-toggle="modal"
    data-bs-target="#categoryModal"
  >
    <i class="bi bi-eye"></i>
    <span class="ms-1">View Categories</span>
  </button>
</div>
<div class="card mb-4">
  <div class="card-body">
    <h6 class="mb-3">Spending Comparison</h6>
    <canvas id="periodCompareChart" height="120"></canvas>
  </div>
</div>

<!-- TABLE -->
<div class="card">
<div class="card-body table-responsive">
<table class="table table-sm table-bordered align-middle">

<thead class="table-light">
<tr>
  <th>Date</th>
  <th>Total</th>
  <th class="text-center">View</th>
</tr>
</thead>

<tbody>
<?php
$currentDate = null;
$dayTotal = 0;
$rows = [];
$i = 0;

while($e = $expenses->fetch_assoc()){

  if($currentDate && $currentDate !== $e['expense_date']){
    $cid = "day$i";
?>
<tr class="table-secondary">
  <td><?= date('M d, Y',strtotime($currentDate)) ?></td>
  <td>₵ <?= number_format($dayTotal,2) ?></td>
  <td class="text-center">
    <button class="btn btn-sm btn-primary" data-bs-toggle="collapse" data-bs-target="#<?= $cid ?>">View</button>
  </td>
</tr>

<tr class="collapse" id="<?= $cid ?>">
<td colspan="3" class="p-0">
<table class="table table-sm mb-0">
<?php foreach($rows as $r): ?>
<tr>
<td><?= htmlspecialchars($r['category']) ?></td>
<td><?= htmlspecialchars($r['description']) ?></td>
<td class="text-end">₵ <?= number_format($r['amount'],2) ?></td>
<td class="text-end">
<button class="btn btn-sm btn-outline-warning edit-expense"
 data-id="<?= $r['id'] ?>"
 data-description="<?= htmlspecialchars($r['description'],ENT_QUOTES) ?>"
 data-amount="<?= $r['amount'] ?>">
<i class="bi bi-pencil"></i>
</button>

<button class="btn btn-sm btn-outline-danger delete-expense"
 data-id="<?= $r['id'] ?>">
<i class="bi bi-trash"></i>
</button>
</td>
</tr>
<?php endforeach; ?>
</table>
</td>
</tr>
<?php
    $rows=[];
    $dayTotal=0;
    $i++;
  }

  $currentDate=$e['expense_date'];
  $dayTotal += $e['amount'];
  $rows[]=$e;
}

if($rows):
$cid="day$i";
?>
<tr class="table-secondary">
<td><?= date('M d, Y',strtotime($currentDate)) ?></td>
<td>₵ <?= number_format($dayTotal,2) ?></td>
<td class="text-center">
<button class="btn btn-sm btn-primary" data-bs-toggle="collapse" data-bs-target="#<?= $cid ?>">View</button>
</td>
</tr>

<tr class="collapse" id="<?= $cid ?>">
<td colspan="3" class="p-0">
<table class="table table-sm mb-0">
<?php foreach($rows as $r): ?>
<tr>
<td><?= htmlspecialchars($r['category']) ?></td>
<td><?= htmlspecialchars($r['description']) ?></td>
<td class="text-end">₵ <?= number_format($r['amount'],2) ?></td>
<td class="text-end">
<button class="btn btn-sm btn-outline-warning edit-expense"
 data-id="<?= $r['id'] ?>"
 data-description="<?= htmlspecialchars($r['description'],ENT_QUOTES) ?>"
 data-amount="<?= $r['amount'] ?>">
<i class="bi bi-pencil"></i>
</button>

<button class="btn btn-sm btn-outline-danger delete-expense"
 data-id="<?= $r['id'] ?>">
<i class="bi bi-trash"></i>
</button>
</td>
</tr>
<?php endforeach; ?>
</table>
</td>
</tr>
<?php endif; ?>
</tbody>
</table>
</div>
</div>

</div>
<!-- CATEGORY MODAL -->
<div class="modal fade" id="categoryModal" tabindex="-1">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">

      <div class="modal-header">
        <h6 class="modal-title">Category Breakdown</h6>
        <button class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <div class="modal-body">
        <?php if ($catSummary->num_rows === 0): ?>
          <div class="text-center text-muted">No data</div>
        <?php else: ?>
          <?php while ($c = $catSummary->fetch_assoc()): ?>
            <div class="d-flex justify-content-between border-bottom py-2">
              <span><?= htmlspecialchars($c['name']) ?></span>
              <strong>₵ <?= number_format($c['used'],2) ?></strong>
            </div>
          <?php endwhile; ?>
        <?php endif; ?>
      </div>

    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
const ctx = document.getElementById('periodCompareChart');

new Chart(ctx, {
  type: 'bar',
  data: {
    labels: [
      'Current (<?= $currentLabel ?>)',
      'Previous (<?= $previousLabel ?>)'
    ],
    datasets: [{
      label: 'Amount Spent (₵)',
      data: [
        <?= $usedTotal ?>,
        <?= $prevTotal ?>
      ],
      borderWidth: 1
    }]
  },
  options: {
    responsive: true,
    plugins: {
      legend: { display: false },
      tooltip: {
        callbacks: {
          label: ctx => '₵ ' + ctx.raw.toFixed(2)
        }
      }
    },
    scales: {
      y: {
        beginAtZero: true
      }
    }
  }
});
</script>

<script>
document.getElementById('filterForm').addEventListener('change',()=>filterForm.submit());

document.addEventListener('click',function(e){

if (e.target.closest('.delete-expense')) {
  const id = e.target.closest('.delete-expense').dataset.id;

  Swal.fire({
    title: 'Delete expense?',
    text: 'This action cannot be undone',
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#d33',
    confirmButtonText: 'Yes, delete'
  }).then(result => {

    if (!result.isConfirmed) return;

    fetch('expenses/expense_delete_ajax.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded'
      },
      body: 'id=' + encodeURIComponent(id)
    })
    .then(res => res.json())
   .then(data => {
  if (data.success) {
    Swal.fire('Deleted', data.message, 'success')
      .then(() => location.reload());
  } else {
    Swal.fire('ERROR', data.message, 'error'); // <-- IMPORTANT
  }
})

    .catch(() => {
      Swal.fire(
        'Error',
        'Server error occurred',
        'error'
      );
    });

  });
}

if (e.target.closest('.edit-expense')) {
  const b = e.target.closest('.edit-expense');

  Swal.fire({
    title: 'Edit Expense',
    html: `
      <input id="sw-desc" class="form-control mb-2" value="${b.dataset.description}">
      <input id="sw-amt" type="number" step="0.01" class="form-control" value="${b.dataset.amount}">
    `,
    showCancelButton: true,
    confirmButtonText: 'Save',
    preConfirm: () => ({
      description: document.getElementById('sw-desc').value,
      amount: document.getElementById('sw-amt').value
    })
  }).then(r => {
    if (!r.isConfirmed) return;

    fetch('expenses/expense_update.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded'
      },
      body:
        'id=' + encodeURIComponent(b.dataset.id) +
        '&description=' + encodeURIComponent(r.value.description) +
        '&amount=' + encodeURIComponent(r.value.amount)
    })
    .then(res => res.json())
    .then(data => {
      if (data.success) {
        Swal.fire('Updated', data.message, 'success')
          .then(() => location.reload());
      } else {
        Swal.fire('Error', data.message, 'error');
      }
    });
  });
}


});
</script>

</body>
</html>
