<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

require "config/db.php";
$user_id = (int)$_SESSION['user_id'];

/* =========================
   PAGE TITLE
========================= */
$pageTitle = "Planned Expenses";

/* =========================
   FILTER INPUTS
========================= */
$period = $_GET['period'] ?? 'month';
$month  = $_GET['month']  ?? date('m');
$year   = $_GET['year']   ?? date('Y');
$week   = $_GET['week']   ?? '';

/* =========================
   DATE RANGE & PLAN TYPE
========================= */
if ($period === 'day') {
    $planType = 'day';
    $from = $to = date('Y-m-d');
} elseif ($period === 'week') {
    $planType = 'week';

    // Generate weeks in selected month
    $weeks = [];
    $start = new DateTime("$year-$month-01");
    $end   = new DateTime($start->format('Y-m-t'));
    while ($start <= $end) {
        $w = $start->format("W");
        $weeks[$w] = $start->format("Y-m-d");
        $start->modify('+7 days');
    }

    if ($week === '') $week = array_key_first($weeks);

    $dto = new DateTime();
    $dto->setISODate($year, $week);
    $from = $dto->format('Y-m-d');
    $dto->modify('+6 days');
    $to = $dto->format('Y-m-d');

} else { // month
    $planType = 'month';
    $from = "$year-$month-01";
    $to   = date('Y-m-t', strtotime($from));
}

/* =========================
   FETCH PLANNED EXPENSES
========================= */
$sql = "
SELECT
    p.category_id,
    IFNULL(c.name,'Uncategorized') AS category_name,
    SUM(p.planned_amount) AS planned_amount,
    (
        SELECT IFNULL(SUM(e.amount),0)
        FROM expenses e
        WHERE e.user_id = p.user_id
          AND e.category_id = p.category_id
          AND e.expense_date BETWEEN ? AND ?
    ) AS used_amount
FROM planned_expenses p
LEFT JOIN categories c ON c.id = p.category_id
WHERE p.user_id = ?
  AND p.plan_type = ?
  AND p.planned_date BETWEEN ? AND ?
GROUP BY p.category_id, c.name
ORDER BY category_name ASC
";

$stmt = $conn->prepare($sql);
$stmt->bind_param(
    "ssisss",
    $from,
    $to,
    $user_id,
    $planType,
    $from,
    $to
);
$stmt->execute();
$res = $stmt->get_result();

/* =========================
   PROCESS DATA & KPI
========================= */
$rows = [];
$totalPlanned   = 0;
$totalUsed      = 0;
$totalRemaining = 0;
$totalOverUsed  = 0;

while ($r = $res->fetch_assoc()) {
    $planned = (float)$r['planned_amount'];
    $used    = (float)$r['used_amount'];
    $totalPlanned += $planned;
    $totalUsed    += $used;

    if ($used > $planned) {
        $status = 'Over Used';
        $totalOverUsed += ($used - $planned);
    } elseif ($used == $planned) {
        $status = 'Achieved';
    } else {
        $status = 'Pending';
        $totalRemaining += ($planned - $used);
    }

    $r['status'] = $status;
    $rows[] = $r;
}

$stmt->close();

/* =========================
   MONTH LIST
========================= */
$months = [
    '01'=>'January','02'=>'February','03'=>'March','04'=>'April',
    '05'=>'May','06'=>'June','07'=>'July','08'=>'August',
    '09'=>'September','10'=>'October','11'=>'November','12'=>'December'
];
?>

<?php require "includes/head.php"; ?>
<?php include('includes/preloader.php'); ?>
<?php require "includes/navbar.php"; ?>

<div class="container my-4">

<h4 class="mb-3">
    <i class="bi bi-calendar-check"></i> <?= $pageTitle ?>
</h4>

<!-- FILTER -->
<form class="row g-2 mb-4">
    <div class="col-md-2">
        <select name="period" class="form-select form-select-sm" onchange="this.form.submit()">
            <option value="day"   <?= $period==='day'?'selected':'' ?>>Today</option>
            <option value="week"  <?= $period==='week'?'selected':'' ?>>Week</option>
            <option value="month" <?= $period==='month'?'selected':'' ?>>Month</option>
        </select>
    </div>

    <div class="col-md-3">
        <select name="month" class="form-select form-select-sm" onchange="this.form.submit()">
            <?php foreach($months as $num=>$name): ?>
                <option value="<?= $num ?>" <?= $month==$num?'selected':'' ?>><?= $name ?> <?= $year ?></option>
            <?php endforeach; ?>
        </select>
    </div>

    <?php if($period==='week'): ?>
        <div class="col-md-2">
            <select name="week" class="form-select form-select-sm" onchange="this.form.submit()">
                <?php foreach($weeks as $w=>$startDate): ?>
                    <option value="<?= $w ?>" <?= $week==$w?'selected':'' ?>>
                        Week <?= $w ?> (<?= date('M d', strtotime($startDate)) ?>)
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
    <?php endif; ?>
</form>

<!-- KPI -->
<div class="row g-3 mb-4">
    <div class="col-md-3">
        <div class="p-3 rounded text-white" style="background:#0ea5e9">
            <i class="bi bi-wallet2"></i> Total Planned
            <h4>₵ <?= number_format($totalPlanned,2) ?></h4>
        </div>
    </div>

    <div class="col-md-3">
        <div class="p-3 rounded text-white" style="background:#16a34a">
            <i class="bi bi-check-circle"></i> Used
            <h4>₵ <?= number_format($totalUsed,2) ?></h4>
        </div>
    </div>

    <div class="col-md-3">
        <div class="p-3 rounded text-white" style="background:#f59e0b">
            <i class="bi bi-hourglass-split"></i> Remaining
            <h4>₵ <?= number_format($totalRemaining,2) ?></h4>
        </div>
    </div>

    <div class="col-md-3">
        <div class="p-3 rounded text-white" style="background:#dc2626">
            <i class="bi bi-exclamation-triangle"></i> Over Used
            <h4>₵ <?= number_format($totalOverUsed,2) ?></h4>
        </div>
    </div>
</div>

<!-- TABLE -->
<div class="card">
<div class="card-body table-responsive">

<table class="table table-sm table-bordered align-middle">
<thead class="table-light">
<tr>
<th>Category</th>
<th>Planned</th>
<th>Used</th>
<th>Status</th>
<th>Action</th>
</tr>
</thead>
<tbody>

<?php if(empty($rows)): ?>
<tr><td colspan="5" class="text-center text-muted">No planned expenses found</td></tr>
<?php else: foreach($rows as $r): ?>
<tr>
<td><?= htmlspecialchars($r['category_name']) ?></td>
<td>₵ <?= number_format($r['planned_amount'],2) ?></td>
<td>₵ <?= number_format($r['used_amount'],2) ?></td>
<td>
<?php if($r['status']=='Achieved'): ?>
<span class="badge bg-success">Achieved</span>
<?php elseif($r['status']=='Over Used'): ?>
<span class="badge bg-danger">Over Used</span>
<?php else: ?>
<span class="badge bg-warning text-dark">Pending</span>
<?php endif; ?>
</td>
<td class="text-center">
    <button 
        class="btn btn-sm btn-outline-primary edit-plan" 
        data-id="<?= $r['category_id'] ?>" 
        data-amount="<?= $r['planned_amount'] ?>"
    >
        <i class="bi bi-pencil"></i> Edit
    </button>
</td>
</tr>
<?php endforeach; endif; ?>

</tbody>

</table>
</div>
</div>

</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
document.addEventListener('click', function(e){
    if(e.target.closest('.edit-plan')){
        const btn = e.target.closest('.edit-plan');
        const planId = btn.dataset.id;
        const amount = btn.dataset.amount;

        Swal.fire({
            title: 'Edit Planned Amount',
            html: `<input id="sw-amount" type="number" class="form-control" value="${amount}" min="0" step="0.01">`,
            showCancelButton: true,
            confirmButtonText: 'Save',
            preConfirm: () => {
                return document.getElementById('sw-amount').value;
            }
        }).then(result => {
            if(!result.isConfirmed) return;

            const newAmount = result.value;

            fetch('expenses/planned_update.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'id=' + encodeURIComponent(planId) + '&amount=' + encodeURIComponent(newAmount)
            })
            .then(res => res.json())
            .then(data => {
                if(data.success){
                    Swal.fire('Updated', data.message, 'success').then(()=>location.reload());
                } else {
                    Swal.fire('Error', data.message, 'error');
                }
            })
            .catch(()=> Swal.fire('Error', 'Server error', 'error'));
        });
    }
});
</script>
<?php $conn->close(); ?>
</body>
</html>
